#include "hardware.h"
#include "types.h"

volatile BYTE Flags = 0;

void InitSystem(void)
{
  TCCR2 = (1<<WGM21)|(0<<WGM20)|(1<<CS22)|(1<<CS21)|(0<<CS20);
  OCR2 = 249; // CTC mode, 250 Hz
  TCNT2 = 0;
  TIMSK = (1<<OCIE2);
  TIFR |= (1<<OCF2); // clear pending interrupt
}

#define     ST_WAIT_KEY     0
#define     ST_CHECK_KEY    1
#define     ST_RELEASE_WAIT 2

static BYTE KeyCode = 0;
static BYTE ScanState = ST_WAIT_KEY;
static BYTE CheckedKey;
static BYTE CheckKeyCnt;
static BYTE PressCnt;

#define PRESS_CNT      1
#define RELEASE_CNT    2
#define LONG_PRESS_CNT 200

void ScanKbd(void)
{                     
BYTE C = GETKEYCODE;
  switch(ScanState)
    {
    case ST_WAIT_KEY:
      if(C != 0)
        {
        CheckedKey = C;
        CheckKeyCnt = PRESS_CNT;
        ScanState = ST_CHECK_KEY;
        }
      break;

    case ST_CHECK_KEY:
      if(C == CheckedKey)
        {
        CheckKeyCnt--;
        if(!CheckKeyCnt)
          {
          SETFLAG(F_Kbd);
          KeyCode = CheckedKey;
          ScanState = ST_RELEASE_WAIT;
          CheckKeyCnt = RELEASE_CNT;
          PressCnt = LONG_PRESS_CNT;
          }
        }
      else
        ScanState = ST_WAIT_KEY;
      break;

    case ST_RELEASE_WAIT:
      if(C != 0)
        {
        CheckKeyCnt = RELEASE_CNT;
        if(PressCnt)
          PressCnt--;
        else
          {
          SETFLAG(F_Kbd);
          KeyCode |= K_LPR; // long press indication
          }
        }
      else
        {
        CheckKeyCnt--;    
        if(!CheckKeyCnt)         
          {
          ScanState = ST_WAIT_KEY;
          SETFLAG(F_Kbd);
          KeyCode &= ~K_LPR;
          KeyCode |= K_RLS; // release indication
          }
        }
      break;
  }
}

BYTE GetKey(void)
{
  return KeyCode;
}

static volatile BYTE Cnt = 0;
static volatile BYTE Delay = 0;

#pragma vector = TIMER2_COMP_vect
__interrupt void Timer2(void)
{
  Cnt++;
  if(Cnt == 25)
    { // 10 Hz
    Cnt = 0;
    SETFLAG(F_Tick);
    if(Delay) Delay--;
    }
  ScanKbd();
}

void Pause(BYTE T)
{
  Delay = T;
  while(Delay);
}

