#include "types.h"
#include "hardware.h"
#include "sys.h"    

#define F_FirstFlash 0x01
#define F_Window     0x02

// prescaler: 16000000 / 256 = 62500 Hz
#define PRESCALER   (1<<CS12)|(0<<CS11)|(0<<CS10)
#define MAXPPM 60000

BYTE  BladeCnt = 2;

volatile WORD RPM;
volatile WORD PPM_period;

static WORD FlashCnt;
static BYTE FTacho;

__monitor void InitRPM(void)
{
  FTacho = F_FirstFlash;
  RPM = 0;
  MCUCR |= (1<<ISC11)|(0<<ISC10);
  GIFR |= (1<<INTF1);
  GICR |= (1<<INT1);
  OCR1B = 12500;      
  OCR1A = 16000;
  TCNT1 = 0;
  TCCR1A = 0;
  TCCR1B = PRESCALER;
  TIMSK |= ((1<<OCIE1A)|(1<<OCIE1B));
}

__monitor void DoneRPM(void)
{
  GICR = 0;
  TIMSK &= ~((1<<OCIE1A)|(1<<OCIE1B));
  TCCR1B = 0;
}

#pragma vector = INT1_vect
__interrupt void RPM_Pulse(void)
{
  DWORD T;
  if(FTacho & F_FirstFlash)
    {
    FTacho = 0;
    FlashCnt = 0;
    // timer1 setup compA,B enabled
    TCNT1 = 0;
    TIFR |= ((1<<OCF1A)|(1<<OCF1B)); // clear pending interrupt
    TIMSK |= ((1<<OCIE1A)|(1<<OCIE1B));
    SFIOR |= 1<<PSR10;
    TCCR1B = PRESCALER; // start      
    }
  else
    {
    FlashCnt++;
    if(FTacho & F_Window)
      {
      TCCR1B = 0; // Stop timer
      TIMSK &= ~((1<<OCIE1B)|(1<<TOIE1)); // disable timer interrupt
      // calculate
      T = (62500L * 60L * (DWORD)FlashCnt);
      T /= TCNT1;
      RPM = T / BladeCnt;
      FTacho = F_FirstFlash; // prepare the next cycle
      GIFR |= (1<<INTF1); // clear pending interrupt 
      SETFLAG(F_Update);
      }
    }
}

// time window
#pragma vector = TIMER1_COMPB_vect
__interrupt void TimeWindow(void)
{
  FTacho |= F_Window;
}

static volatile WORD start_time;
static BYTE i=0;
static volatile WORD PW[8];

// timeout
#pragma vector=TIMER1_COMPA_vect
__interrupt void TIMER1_COMPA(void)
{
  if(TCCR1B & (1<<ICNC1))
    {
    OCR1A += MAXPPM;
    PPM_period = 0xFFFF;
    for(BYTE i=0; i<8; i++) PW[i]=0;
    }
  else
    {
    RPM = 0;
    FTacho = F_FirstFlash;
    }
  SETFLAG(F_Update);
}

#pragma vector=TIMER1_CAPT_vect
__interrupt void TIMER1_CAPT(void)
{
  WORD icr;
  BYTE tccr1b;
  icr = ICR1;                     // capture timestamp
  tccr1b = TCCR1B;
  TCCR1B = tccr1b ^ (1 << ICES1); // reverse sense
  if((tccr1b & (1 << ICES1)) == (1 << ICES1))
    {
    PPM_period = (icr - start_time)/2; // Length of previous period
    start_time = icr;              // Start of new pulse/period
    OCR1A = icr + MAXPPM;               // Move timeout window
    TIFR |= (1 << OCF1A);          // Clear in case of race
    }
  else
    {
    PW[i++] = icr - start_time;      // Capture falling-edge time
    i &= 0x07;
    if(i == 0) SETFLAG(F_Update);
    }
}

WORD PPM_pulse_width(void)
{
WORD S = 0;
for(BYTE i=0; i<8; i++) S += PW[i];
return S >>= 4;
}

__monitor void InitICP(void)
{
  start_time = 0;
  PPM_period = 0xFFFF;
  TCCR1A = 0;
  TCCR1B = (1<<ICNC1)|(1 << ICES1)|((0 << CS12)|(1 << CS11)|(0 << CS10)); // XTAL/8=2MHz
  TCNT1 = 0;
  OCR1A = MAXPPM;
  GICR = 0;
  GIFR |= (1<<INTF1); // clear pending interrupts
  TIMSK &= ~(1 << OCIE1B);
  TIFR |= (1<<OCF1A)|(1<<OCF1B); // clear pending interrupts
  TIMSK |= (1 << OCIE1A)|(1 << TICIE1);
}

__monitor void DoneICP(void)
{
  TIMSK &= ~((1 << OCIE1A)|(1 << TICIE1));
  TCCR1B = 0;
}
