{ Multi-Setting  - Greg Glenn
   Store/retrieve multiple sets of parameters
   Built on Settings object by Robert Quattlebaum

   The Settings object allows data settings to be stored
   in eeprom by a two-character key.  This object takes it
   one step further by allowing you to store multiple lists of
   settings, each list identified by a primary key that is one
   of the two characters in the settings key.  Settings within
   each list are identified by the primary key plus the subkey.

   From the Settings object viewpoint, this is all just one flat
   set of data.

   This object will let you store up to 26 lists,
   but this could be expanded beyond the A-Z primary key bounds
   if desired.

Alternate way of doing this:

CON
  MAX_NAME_SIZE = 20
  MAX_ROOMS     = 15
  MAX_DEVICES   = 50
 
DAT
  RoomNames     byte 0[MAX_NAME_SIZE*MAX_ROOMS]
  DeviceNames   byte 0[MAX_NAME_SIZE*MAX_DEVICES]
  DeviceRooms   byte 0]MAX_DEVICES]
  DeviceTypes   byte 0[MAX_DEVICES]
  DeviceNumbers byte 0[MAX_DEVICES]
  DeviceStates  byte 0[MAX_DEVICES]
 
The names should be null-terminated, and you would access entries as follows:
 
  Room := DeviceRooms[Index]
  Type := DeviceTypes[Index]
  Number := DeviceNumbers[Index]
  State := DeviceStates[Index]
  DeviceName := @DeviceNames + (Index * MAX_NAME_SIZE)
  RoomName := @RoomNames + (Room * MAX_NAME_SIZE)

}
 

CON

  SERIAL_TX_PIN        = 30
  SERIAL_RX_PIN        = 31
  MODEL_NAME_LENGTH    = 15
  
  ACTIVE_MODEL_KEY_KEY  = "0"+("0"<<8)
  MODEL_NAME_SUBKEY         = ("9"<<8)
  ANGULAR_GAIN_SUBKEY       = ("a"<<8)   
  RATE_GAIN_SUBKEY          = ("b"<<8)   
  ANGULAR_DECAY_SUBKEY      = ("c"<<8)   
  PHASE_ANGLE_SUBKEY        = ("d"<<8) 
  SERVO_1_REVERSE_SUBKEY    = ("e"<<8)
  SERVO_2_REVERSE_SUBKEY    = ("f"<<8)
  SERVO_3_REVERSE_SUBKEY    = ("g"<<8)
  SERVO_1_THETA_SUBKEY      = ("h"<<8)
  SERVO_2_THETA_SUBKEY      = ("i"<<8)
  SERVO_3_THETA_SUBKEY      = ("j"<<8)
  PITCH_GYRO_REVERSE_SUBKEY = ("k"<<8)
  ROLL_GYRO_REVERSE_SUBKEY  = ("l"<<8)
  PULSE_INTERVAL_SUBKEY     = ("m"<<8)
  RX_AUX_ACTIVE_SUBKEY      = ("n"<<8)
  GYRO_X_AXIS_SUBKEY        = ("o"<<8)
  GYRO_Y_AXIS_SUBKEY        = ("p"<<8)
  
OBJ
 parameters     :  "settings"
 serio          :  "FullDuplexSerialPlus"

VAR
  long  rxAuxActive
  long  activeModelKey, lastModelKey
  long  angularGain, rateGain, angularDecay
  long  phaseAngle, pulseInterval 
  long  reversePitchGyro, reverseRollGyro
  long  servo1theta, servo2theta, servo3theta   
  long  servo1reverse, servo2reverse, servo3reverse
  long  gyroXAxisAddress, gyroYAxisAddress 
  long  filteredRollRateAddress, filteredPitchRateAddress
 
PUB Start(fRA, fPA)

  filteredRollRateAddress := fRA
  filteredPitchRateAddress := fPA
  
  parameters.Start
  Initialize
  RetrieveModelParameters

PUB Stop
      serio.stop
      parameters.stop

PUB getAngularGain
  return angularGain
  
PUB getRateGain
  return rateGain

PUB getAngularDecay
  return angularDecay

PUB getRxAuxActive
  return rxAuxActive

PUB getPhaseAngle
  return phaseAngle

PUB getPulseInterval
  return pulseInterval

PUB getReversePitchGyro
  return reversePitchGyro

PUB getReverseRollGyro
  return reverseRollGyro

PUB getServo1Theta
  return servo1Theta

PUB getServo2Theta
  return servo2Theta

PUB getServo3Theta
  return servo3Theta

PUB getServo1Reverse
  return servo1Reverse 

PUB getServo2Reverse 
  return servo2Reverse 

PUB getServo3Reverse 
  return servo3Reverse 

PUB getGyroXAxisAddress
  return gyroXAxisAddress

PUB getGyroYAxisAddress
  return gyroYAxisAddress
    
PUB Initialize
  
  'Set parameters to defaults
  '----------------------------------------------------
  '  Settings for various helis, assumes unit is mounted
  '  top-up and 4-pin connector facing to the left.
  '
  '                 Honeybee    FireFox
  '                --------------------------------------
  'angularGain      := 63       63   
  'rateGain         := 21       21                          
  'angularDecay     := 75       65
  'phaseAngle       := -45      0
  'pulseInterval    := 10       10
  'reversePitchGyro := FALSE    FALSE
  'reverseRollGyro  := TRUE     TRUE
  'rxAuxActive      := FALSE    TRUE
  'servo1reverse    := TRUE     TRUE
  'servo2reverse    := FALSE    FALSE
  'servo3reverse    := FALSE    FALSE
  'servo1theta      := 0        -60
  'servo2theta      := -270     -180
  'servo3theta      := 0        -300
  '----------------------------------------------------

  activeModelKey   := "A"       'Can have A thru Z models, but default to A 
  angularGain      := 63        ' 0 to 100 ?
  rateGain         := 21        ' 0 to 100 ?
  angularDecay     := 50         ' limit 1 to 300 percent 
  phaseAngle       := 0         ' limit from -90 to +90
  pulseInterval    := 10                     ' servo pulse interval
  gyroXAxisAddress := filteredRollRateAddress      ' assign axes based on how the unit is 
  gyroYAxisAddress := filteredPitchRateAddress     ' oriented in the aircraft
  reversePitchGyro := FALSE
  reverseRollGyro  := TRUE
  servo1reverse    := TRUE
  servo2reverse    := FALSE
  servo3reverse    := FALSE
  servo1theta      := -60       'Negative angles are clockwise from nose=0 when looking
  servo2theta      := -180      ' down on rotor mast
  servo3theta      := -300
  
  UpdateLongFromEeprom( @activeModelKey, ACTIVE_MODEL_KEY_KEY )


PUB  AllowUserToEdit(statusLEDPin)
  '-----------------------------------------------------------------------------
  'If using PropStick USB, insert a check to see if USB/FTDI is connected and powered
  ' Pins 30 and 31 should be High if they are
  ' Can't start serial io if cable disconnected because processor
  '  will hang or do weird things
  ' For a standalone Propeller chip being programmed
  ' via a PropPlug, when the PropPlug is disconnected, the tx/rx pins
  ' float and starting the serial io cog does not seem to hang the
  ' processor.  in that case, this check may or may not see rx/tx as high,

  '  if((ina[SERIAL_RX_PIN] == 1) and (ina[SERIAL_TX_PIN] == 1)) 
  '-------------------------------------------------------------------------------

  dira[SERIAL_TX_PIN]~  'Set for input

    if(ina[SERIAL_TX_PIN] == 1)
      serio.start(SERIAL_RX_PIN, SERIAL_TX_PIN, 0, 57600) 'Start cog to allow IO with serial terminal

      outa[statusLEDpin]~~          'Turn on LED to let user know we are ready for his input

      serio.rxflush
      waitcnt(clkfreq/500+cnt)
  
      if(serio.rxtime(5000)<> -1)   'Allow a few seconds for some input from terminal, indicating the                              
         EditParameters            ' user wants to edit parameters, otherwise skip edit step 
          
      outa[statusLEDpin]~           'Turn off LED to let user know we no longer accept input
      waitcnt(clkfreq/50+cnt)      ' Wait a bit for any serial output to flush
      serio.stop

  
PRI UpdateGyroAxisFromEeprom(addr, default, key) | tempstr[2]

  if(parameters.findKey(key))
    parameters.getString(key,@tempstr,2)
    if(tempstr == "R")
      long[addr] := filteredRollRateAddress  
    if(tempstr == "P")
      long[addr] := filteredPitchRateAddress       
  else
      parameters.setString(key, default)
        
PRI UpdateLongFromEeprom(addr, key)

  if(parameters.findKey(key))
    long[addr] :=  parameters.getLong(key)
  else
    parameters.setLong(key, long[addr])

PRI UpdateStringFromEeprom(addr, key)

  if(parameters.findKey(key))
    parameters.getString(key,addr,MODEL_NAME_LENGTH)
  else
    parameters.setString(key, string("unknown"))
    parameters.getString(key,addr,MODEL_NAME_LENGTH)

PRI  RetrieveModelParameters
  
  UpdateStringFromEeprom(   @modelName,             (activeModelKey + MODEL_NAME_SUBKEY)          ) 
  UpdateLongFromEeprom(     @angularGain,           (activeModelKey + ANGULAR_GAIN_SUBKEY)        )
  UpdateLongFromEeprom(     @rateGain,              (activeModelKey + RATE_GAIN_SUBKEY)           )
  UpdateLongFromEeprom(     @angularDecay,          (activeModelKey + ANGULAR_DECAY_SUBKEY)       )
  UpdateLongFromEeprom(     @phaseAngle,            (activeModelKey + PHASE_ANGLE_SUBKEY)         )
  UpdateLongFromEeprom(     @reversePitchGyro,      (activeModelKey + PITCH_GYRO_REVERSE_SUBKEY)  )
  UpdateLongFromEeprom(     @reverseRollGyro,       (activeModelKey + ROLL_GYRO_REVERSE_SUBKEY)   )
  UpdateLongFromEeprom(     @pulseInterval,         (activeModelKey + PULSE_INTERVAL_SUBKEY)      )
  'UpdateLongFromEeprom(     @rxAuxActive,           (activeModelKey + RX_AUX_ACTIVE_SUBKEY)       )
  UpdateLongFromEeprom(     @servo1theta,           (activeModelKey + SERVO_1_THETA_SUBKEY)       )
  UpdateLongFromEeprom(     @servo2theta,           (activeModelKey + SERVO_2_THETA_SUBKEY)       )
  UpdateLongFromEeprom(     @servo3theta,           (activeModelKey + SERVO_3_THETA_SUBKEY)       ) 
  UpdateLongFromEeprom(     @servo1reverse,         (activeModelKey + SERVO_1_REVERSE_SUBKEY)     )
  UpdateLongFromEeprom(     @servo2reverse,         (activeModelKey + SERVO_2_REVERSE_SUBKEY)     )
  UpdateLongFromEeprom(     @servo3reverse,         (activeModelKey + SERVO_3_REVERSE_SUBKEY)     )
  UpdateGyroAxisFromEeprom( @gyroXAxisAddress, "R", (activeModelKey + GYRO_X_AXIS_SUBKEY)     )
  UpdateGyroAxisFromEeprom( @gyroYAxisAddress, "P", (activeModelKey + GYRO_Y_AXIS_SUBKEY)     )
           
PRI EditParameters | response
  '---------------------------------------------------------------------------------
  ' Use Robert Quattlebaum's Settings object to store/retrieve parameters in eeprom
  '  allowing user to edit them via serial/USB to serial terminal tool
  '---------------------------------------------------------------------------------
  repeat
    serio.tx($D)
    serio.tx($D)
    serio.str(string("Parameters:"))       'Dump all parameters    
    serio.tx($D)
    serio.str(string("Model: "))
    serio.tx(activeModelKey)
    serio.str(string(": "))
    serio.str(@modelName)
    serio.tx($D)
    DumpInteger(string("Rate Gain"),                    @rateGain)
    DumpInteger(string("Angular Gain"),                 @angularGain)
    DumpInteger(string("Angular Decay"),                @angularDecay)
    DumpInteger(string("PhaseAngle"),                   @phaseAngle)
    DumpInteger(string("Servo Pulse Interval"),         @pulseInterval)
    DumpAxis(string("Gyro X Axis"),                     @gyroXAxisAddress)
    DumpAxis(string("Gyro Y Axis"),                     @gyroYAxisAddress)
    DumpSwitch(string("Reverse Pitch Gyro"),            @reversePitchGyro)        
    DumpSwitch(string("Reverse Roll Gyro"),             @reverseRollGyro)
    'DumpSwitch(string("RX AUX channel active"),         @rxAuxActive)
    DumpInteger(string("Servo 1 Theta"),                @servo1theta)
    DumpInteger(string("Servo 2 Theta"),                @servo2theta)
    DumpInteger(string("Servo 3 Theta"),                @servo3theta)      
    DumpSwitch(string("Reverse Servo 1 "),              @servo1reverse)
    DumpSwitch(string("Reverse Servo 2 "),              @servo2reverse)
    DumpSwitch(string("Reverse Servo 3 "),              @servo3reverse)
      
    serio.str(string("(l)ist, (c)hange, (s)ave (e)xit+save, (q)uit, (m)odel "))
    response := serio.rx                'Get command byte

    if((response) == "q")
       serio.tx($D)
       serio.str(string("Quitting...reverting to last saved model and its parameters "))
       serio.str(string("..."))
       serio.tx($D)
       activeModelKey := lastModelKey 
       parameters.revert                'Revert parms from eeprom
       RetrieveModelParameters
       serio.str(string("Active Model is now: "))
       serio.str(@activeModelKey)
       serio.str(string(": "))
       serio.str(@modelName)        
       serio.tx($D)
       serio.str(string("Done."))
       serio.tx($D)
       QUIT
       
    if((response) == "l")               'Go back to start of loop
       NEXT
       
    if((response) == "s")               'Save to eeprom
       serio.tx($D)
       serio.str(string("Saving parameters to eeprom..."))      
       parameters.commit
       lastModelKey := activeModelKey
       serio.str(string("Done."))
       serio.tx($D)
       NEXT
       
    if((response) == "m")
       serio.tx($D)
       SelectModel

    if((response) == "e")
      serio.str(string("Saving parameters to eeprom..."))      
       parameters.commit
       serio.str(string("Done."))
       serio.tx($D)
       QUIT
      
    if((response) == "c")
      EditString (string("Model Name"),            @modelName,            (activeModelKey + MODEL_NAME_SUBKEY)           )
      EditInteger(string("Rate Gain"),             @rateGain,1,100,       (activeModelKey + RATE_GAIN_SUBKEY)            )
      EditInteger(string("Angular Gain"),          @angularGain,1,100,    (activeModelKey + ANGULAR_GAIN_SUBKEY)         )
      EditInteger(string("Angular Decay"),         @angularDecay,1,300,   (activeModelKey + ANGULAR_DECAY_SUBKEY)        )
      EditInteger(string("Phase Angle"),           @phaseAngle,-90,90,    (activeModelKey + PHASE_ANGLE_SUBKEY)          )         
      EditInteger(string("Servo Pulse Interval"),  @pulseInterval,5,20,   (activeModelKey + PULSE_INTERVAL_SUBKEY)       )
      EditAxis   (string("Gyro X Axis"),           @gyroXAxisAddress,     (activeModelKey + GYRO_X_AXIS_SUBKEY)          )
      EditAxis   (string("Gyro Y Axis"),           @gyroYAxisAddress,     (activeModelKey + GYRO_Y_AXIS_SUBKEY)          )
              
      EditSwitch (string("Reverse Pitch Gyro"),    @reversePitchGyro,     (activeModelKey + PITCH_GYRO_REVERSE_SUBKEY)   )
      EditSwitch (string("Reverse Roll Gyro"),     @reverseRollGyro,      (activeModelKey + ROLL_GYRO_REVERSE_SUBKEY)    )
      'EditSwitch (string("RX Aux channel active"), @rxAuxActive,          (activeModelKey + RX_AUX_ACTIVE_SUBKEY)        )
      EditInteger(string("Servo 1 Theta"),         @servo1theta,-360,360, (activeModelKey + SERVO_1_THETA_SUBKEY)        )
      EditInteger(string("Servo 2 Theta"),         @servo2theta,-360,360, (activeModelKey + SERVO_2_THETA_SUBKEY)        )
      EditInteger(string("Servo 3 Theta"),         @servo3theta,-360,360, (activeModelKey + SERVO_3_THETA_SUBKEY)        )      
      EditSwitch (string("Reverse Servo 1 "),      @servo1reverse,        (activeModelKey + SERVO_1_REVERSE_SUBKEY)      )
      EditSwitch (string("Reverse Servo 2 "),      @servo2reverse,        (activeModelKey + SERVO_2_REVERSE_SUBKEY)      )
      EditSwitch (string("Reverse Servo 3 "),      @servo3reverse,        (activeModelKey + SERVO_3_REVERSE_SUBKEY)      )      

  return

PRI DumpInteger(label, parmAddr)
  serio.str(label)
  serio.str(string(": ["))
  serio.dec(long[parmAddr])
  serio.str(string("]"))
  serio.tx($D)

PRI DumpSwitch(label, parmAddr)  ' "Y" = reverse = TRUE,  "N" = don't reverse = FALSE
  serio.str(label)
  serio.str(string(": ["))
  
  if(long[parmAddr] == TRUE)
     serio.str(string("Y"))
  else
     serio.str(string("N"))
       
  serio.str(string("]"))
  serio.tx($D)

PRI DumpAxis(label, parmAddr)  ' "P" = pitch,  "R" = roll,  "Y" = yaw
  serio.str(label)
  serio.str(string(": ["))
  
  if(long[parmAddr] == filteredPitchRateAddress)
     serio.str(string("P"))
     
  if(long[parmAddr] == filteredRollRateAddress)
     serio.str(string("R"))   
       
  serio.str(string("]"))
  serio.tx($D)   
   
PRI EditInteger(label, parmAddr,lowVal, highVal, key) | response, tempstr[11]

  serio.tx($D)
  serio.str(label)
  serio.str(string("("))
  serio.dec(lowVal)
  serio.str(string(" to "))
  serio.dec(highVal)
  serio.str(string(")")) 
  serio.str(string(": ["))
  serio.dec(long[parmAddr])
  serio.str(string("] -> "))

  serio.getStr(@tempstr)
  if(byte[@tempstr][0] <> 0)               'If user hit return without entering a value, don't
     response := serio.StrToDec(@tempstr)  ' count it as a zero value
     if((response) => lowVal AND (response =< highVal))
        long[parmAddr] := response
        serio.dec(response)
        parameters.setLong(key,long[parmAddr])

PRI EditString(label, parmAddr, key)| index, tempstr[15]
  serio.tx($D)
  serio.str(label)
  serio.str(string(": ["))
  serio.str(parmAddr)
  serio.str(string("] -> "))
  serio.getStr(@tempstr)
  index~
  if(byte[@tempstr][0] <> 0)               'If user hit return without entering a value, don't count it
       repeat until ((byte[parmAddr][index] := byte[@tempstr][index]) == 0)
          index++
          if(index == 15)
             QUIT
       byte[parmAddr][index]~
       serio.str(parmAddr)
       parameters.setString(key,parmAddr) 

PRI EditAxis(label, parmAddr, key) | response, tempstr[11]

  serio.tx($D)
  serio.str(label)
  serio.str(string("((P)itch, (R)oll)"))
  serio.str(string(": ["))
  if(long[parmAddr] == filteredPitchRateAddress)
     serio.str(string("P"))
  if(long[parmAddr] == filteredRollRateAddress)
     serio.str(string("R"))
     
  serio.str(string("] -> "))

  serio.GetStr(@tempstr)
  serio.str(@tempstr)
  
  if(strcomp(@tempstr,String("P")) or strcomp(@tempstr,String("p")))
    long[parmAddr] := filteredPitchRateAddress
    parameters.setString(key, string("P")) 
  elseif(strcomp(@tempstr,String("R")) or strcomp(@tempstr,String("r"))) 
    long[parmAddr] := filteredRollRateAddress
    parameters.setString(key, string("R")) 
 
  
PRI EditSwitch(label, parmAddr, key) | tempstr[11]

  serio.tx($D)
  serio.str(label)
  serio.str(string("("))
  serio.str(string("Y or N"))
  serio.str(string(")")) 
  serio.str(string(": ["))
  
  if(long[parmAddr] == TRUE)
     serio.str(string("Y"))
  else
     serio.str(string("N"))
     
  serio.str(string("] -> "))
  serio.GetStr(@tempstr)
  serio.str(@tempstr)
  
  if(strcomp(@tempstr,String("Y")) or strcomp(@tempstr,String("y")))
    long[parmAddr] := TRUE
  elseif(strcomp(@tempstr,String("N")) or strcomp(@tempstr,String("n"))) 
    long[parmAddr] := FALSE
    
  parameters.setLong(key,long[parmAddr])    


PRI SelectModel  | response, index, tempstr[MODEL_NAME_LENGTH]
   repeat index from "A" to "Z"
       if(parameters.findKey(index + MODEL_NAME_SUBKEY))
         serio.tx(index)
         serio.str(string(": "))
         parameters.getString((index + MODEL_NAME_SUBKEY), @tempstr, MODEL_NAME_LENGTH )
         serio.str(@tempstr)
         serio.tx($D)
  serio.str(string("Choose Model (A thru Z)-> "))
  response := serio.rx
  if((response => "a") and (response =< "z")) 'Upcase small letters
      response -= 32
  if((response => "A") and (response =< "Z"))
      serio.str(string("Switching to model "))
      serio.tx(response)
      serio.tx($D)
      lastModelKey := activeModelKey
      activeModelKey := response
      parameters.setLong(ACTIVE_MODEL_KEY_KEY,activeModelKey)
      ifnot(parameters.findKey(response + MODEL_NAME_SUBKEY))
         serio.str(string("Initializing new model with current parameters."))
         serio.tx($D)
      RetrieveModelParameters
         
DAT
        ModelName byte "Default        ",0

                    