#include <inttypes.h>
#include <compat/twi.h>


/* define CPU frequency in Mhz here if not defined in Makefile */
#ifndef F_CPU
#define F_CPU 1000000UL
#endif

/* I2C clock in Hz */
#define I2C_SPEED  100000L

uint8_t waitTransmissionI2C();
  
// ************************************************************************************************************
// I2C general functions
// ************************************************************************************************************

void i2c_init(void) {
  TWSR = 0;                                    // no prescaler => prescaler = 1
  TWBR = ((F_CPU / I2C_SPEED) - 16) / 2;       // change the I2C clock rate
  TWCR = 1<<TWEN;                              // enable twi module, no interrupt
}

unsigned char i2c_rep_start(uint8_t address)
{
	TWCR = (1<<TWINT) | (1<<TWSTA) | (1<<TWEN) ; // send REPEAT START condition
	if (waitTransmissionI2C())
		return 1;                       // wait until transmission completed
	TWDR = address;                              // send device address
	TWCR = (1<<TWINT) | (1<<TWEN);
	if (waitTransmissionI2C())                       // wail until transmission completed
		return 1;
	return 0;
}

void i2c_stop(void) {
  TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWSTO);
  //  while(TWCR & (1<<TWSTO));                // <- can produce a blocking state with some WMP clones
}

uint8_t i2c_write(uint8_t data ) {
	TWDR = data;                                 // send data to the previously addressed device
	TWCR = (1<<TWINT) | (1<<TWEN);
	return waitTransmissionI2C();
}

uint8_t i2c_read(uint8_t ack, uint8_t *b) {
  TWCR = (1<<TWINT) | (1<<TWEN) | (ack? (1<<TWEA) : 0);
  if (waitTransmissionI2C())
		return 1;
  *b = TWDR;
  if (!ack) i2c_stop();
  return 0;
}

uint8_t waitTransmissionI2C() {
  uint16_t count = 255;
  while (!(TWCR & (1<<TWINT))) {
    count--;
    if (count==0) {              //we are in a blocking state => we don't insist
      TWCR = 0;                  //and we force a reset on TWINT register
      return 1;
    }
  }
  return 0;
}

uint8_t i2c_read_data(uint8_t addr, uint8_t reg, uint8_t *b, uint8_t size) {
  uint8_t bytes_read = 0;
  if (i2c_rep_start(addr)) // I2C write direction
	return 0;
  if (i2c_write(reg))        // register selection
	return 0;
  if (i2c_rep_start((addr) | 1))  // I2C read direction
	return 0;
  while (size--)
  {
    /* acknowledge all but the final byte */
    if (i2c_read(size > 0, b))
		break;
	b++;
    /* TODO catch I2C errors here and abort */
    bytes_read++;
  }
  return bytes_read;
}


uint8_t i2c_write_data(uint8_t addr, uint8_t reg, uint8_t *b, uint8_t size) {
  uint8_t bytes_wrote = 0;
  if (i2c_rep_start(addr)) // I2C write direction
	return 0;
  if (i2c_write(reg))        // register selection
	return 0;
  while (size--)
  {
    if (i2c_write(*b))
		break;
	b++;
    /* TODO catch I2C errors here and abort */
    bytes_wrote++;
  }
  i2c_stop();
  return bytes_wrote;
}
